#  
#  WordCompounder --- Executable file name: GoMusubi
#  Copyright(C) 2021 Kaoru Sagara and Syugo Nakamura 
#  This software is released under any of the GPL (see the file GPL), the LGPL(see the file LGPL), or the BSD License (see the file BSD).

from unicodedata import normalize
from components.utils.unicode import detect_encoding
from components.compounders import ConsecutiveWordCompounder, CommonWordCompounder
from components.mecab_parser import MecabParser, UnidicMecabParser
from components.exceptions import MecabTaggerValueError, LoadFailureException, SaveFailureException, NotFoundException

import os


def process(config, output_dirpath, mode='consecutive', text_filepath="", ipadic_parse_filepath="", unidic_parse_filepath=""):
  """
  解析処理を実行する関数。

  Args:
      config (ConfigParser): iniファイルを読み込んだConfigParserオブジェクト
      output_dirpath (str): 出力ディレクトリのパス
      mode (str, optional): 生成モード('consecutive', 'common'). Defaults to 'consecutive'.
      text_filepath (str, optional): 入力テキストファイルのパス （指定しない場合は形態素解析結果ファイルをロード） Defaults to "".
      ipadic_parse_filepath (str, optional): IPADicの形態素解析結果ファイルのパス. （指定しない場合は入力テキストから形態素解析結果ファイルを生成） Defaults to "".
      unidic_parse_filepath (str, optional): UniDicの形態素解析結果ファイルのパス. （指定しない場合は入力テキストから形態素解析結果ファイルを生成） Defaults to "".

  Raises:
      LoadFailureException: ファイル読み込みに失敗したときの例外
      NotFoundException: ファイルパスやディレクトリパスが存在しないときの例外
      ValueError: 不正な引数が渡されたときの例外
  """
  # ========================================
  # 設定パラメータの読み込み
  # ========================================
  config_dictionary = config['Dictionary']
  ipadic_sys_dirpath = config_dictionary.get('IPADIC_SYSTEM_DIR')
  ipadic_user_filepath = config_dictionary.get('IPADIC_USER_FILE')
  unidic_sys_dirpath = config_dictionary.get('UNIDIC_SYSTEM_DIR')
  unidic_user_filepath = config_dictionary.get('UNIDIC_USER_FILE')

  config_output = config['Output']
  ipadic_output_filename = config_output.get('IPADIC_OUTPUT_FILENAME')
  unidic_output_filename = config_output.get('UNIDIC_OUTPUT_FILENAME')
  result_output_filename = config_output.get('RESULT_OUTPUT_FILENAME')
  compound_word_filename = config_output.get('COMPOUND_WORD_FILENAME')
  cp932 = config_output.getboolean('CP932')
  
  if cp932:
    output_encoding = 'cp932'
  else:
    output_encoding = 'utf-8'

  input_text = None
  if os.path.isfile(text_filepath):
    # ========================================
    # テキストの読み込み
    # ========================================
    try:
      text_encoding = detect_encoding(text_filepath)
      with open(text_filepath, mode='r', encoding=text_encoding) as f:
        input_text = f.read()
    except FileNotFoundError as e:
      raise LoadFailureException("入力テキストファイルが見つかりません。")
    except Exception as e:
      raise LoadFailureException("入力テキストファイルが開けませんでした。")

  # ========================================
  # 出力フォルダの確認
  # ========================================
  if not os.path.isdir(output_dirpath):
    raise NotFoundException("出力フォルダが見つかりません。")

  # ========================================
  # 合成語生成の実行
  # ========================================
  try:
    if mode == 'consecutive':
      ipadic_mecab_parser = MecabParser(ipadic_sys_dirpath, ipadic_user_filepath)

      if input_text:
        # テキストを解析
        ipadic_mecab_parser.parse(input_text, normalize=True)
      elif ipadic_parse_filepath:
        if os.path.isfile(ipadic_parse_filepath):
          # 形態素解析結果をロード
          ipadic_parse_encoding = detect_encoding(ipadic_parse_filepath)
          ipadic_mecab_parser.load(ipadic_parse_filepath, encoding=ipadic_parse_encoding)
        else:
          raise ValueError("形態素解析結果ファイルが見つかりません。")
      else:
        raise ValueError("入力テキストファイルまたは形態素解析結果ファイルを指定してください。")
      
      compounder = ConsecutiveWordCompounder(ipadic_mecab_parser)
    elif mode == 'common':
      ipadic_mecab_parser = MecabParser(ipadic_sys_dirpath, ipadic_user_filepath)
      unidic_mecab_parser = UnidicMecabParser(unidic_sys_dirpath, unidic_user_filepath)

      if input_text:
        # テキストを解析
        ipadic_mecab_parser.parse(input_text, normalize=True)
        unidic_mecab_parser.parse(input_text, normalize=True)
      elif ipadic_parse_filepath and unidic_parse_filepath:
        if os.path.isfile(ipadic_parse_filepath) and os.path.isfile(unidic_parse_filepath):
          # 形態素解析結果をロード
          ipadic_parse_encoding = detect_encoding(ipadic_parse_filepath)
          unidic_parse_encoding = detect_encoding(unidic_parse_filepath)
          ipadic_mecab_parser.load(ipadic_parse_filepath, encoding=ipadic_parse_encoding)
          unidic_mecab_parser.load(unidic_parse_filepath, encoding=unidic_parse_encoding)
        else:
          raise ValueError("形態素解析結果ファイルが見つかりません。")
      else:
        raise ValueError("入力テキストファイルまたは形態素解析結果ファイルを指定してください。")

      compounder = CommonWordCompounder(ipadic_mecab_parser, unidic_mecab_parser)
    else:
      raise ValueError("modeの引数が不正です。")

  except MecabTaggerValueError as e:
    raise
  
  compounder.process()

  # ========================================
  # 各辞書を用いた形態素解析の結果を保存
  # ========================================
  ipadic_output_filepath = os.path.join(output_dirpath, ipadic_output_filename)
  compounder.mecab_parser.save(ipadic_output_filepath, encoding=output_encoding)
  if mode == 'common':
    unidic_output_filepath = os.path.join(output_dirpath, unidic_output_filename)
    compounder.sub_mecab_parser.save(unidic_output_filepath, encoding=output_encoding)

  # ========================================
  # 生成した結合語を保存
  # ========================================
  try:
    compound_word_filepath = os.path.join(output_dirpath, compound_word_filename)
    compounder.save_compound_word(compound_word_filepath, encoding=output_encoding)
  except FileNotFoundError as e:
    SaveFailureException("結合語の保存に失敗しました。")
  
  # ========================================
  # 結合語を用いた形態素解析の結果を保存
  # ========================================
  try:
    result_output_filepath = os.path.join(output_dirpath, result_output_filename)
    compounder.save_result(result_output_filepath, encoding=output_encoding)
  except FileNotFoundError as e:
    SaveFailureException("形態素解析結果を保存できませんでした。")
  